"""Tasks management page."""

import datetime
import streamlit as st
import pandas as pd

from ca_task_manager.services.task_service import TaskService
from ca_task_manager.models.task import Recurrence, Priority
from ca_task_manager.models.user import Role


def render() -> None:
    st.title("Tasks")
    role = st.session_state.get("role")
    # Staff cannot create or edit tasks; they can only view
    can_edit = role in [Role.ADMIN.value, Role.MANAGER.value]
    service = TaskService()
    tasks = service.list_tasks()
    df = pd.DataFrame(tasks)
    if not df.empty:
        df_display = df[["id", "task_code", "title", "client_name", "service_type", "period", "due_date", "priority", "recurrence"]]
        st.dataframe(df_display)
    else:
        st.info("No tasks found.")

    if can_edit:
        st.subheader("Add New Task")
        with st.form("add_task_form"):
            task_code = st.text_input("Task Code", max_chars=20)
            title = st.text_input("Title")
            client_name = st.text_input("Client Name")
            service_type = st.text_input("Service Type")
            period = st.text_input("Period (e.g., FY 2024-25, Q1)")
            due_date = st.date_input("Due Date", min_value=datetime.date.today())
            priority = st.selectbox("Priority", [p.value.capitalize() for p in Priority])
            recurrence = st.selectbox("Recurrence", [r.value.capitalize() for r in Recurrence])
            regulatory_reference = st.text_input("Regulatory Reference")
            notes = st.text_area("Notes")
            submit_task = st.form_submit_button("Add Task")
            if submit_task:
                try:
                    data = {
                        "task_code": task_code.strip(),
                        "title": title.strip(),
                        "client_name": client_name.strip() or None,
                        "service_type": service_type.strip() or None,
                        "period": period.strip() or None,
                        "due_date": due_date,
                        "priority": priority.lower(),
                        "recurrence": recurrence.lower(),
                        "regulatory_reference": regulatory_reference.strip() or None,
                        "notes": notes.strip() or None,
                    }
                    new_id = service.add_task(data)
                    st.success(f"Task created with ID {new_id}.")
                    st.experimental_rerun()
                except Exception as e:
                    st.error(str(e))

        st.subheader("Update Task")
        if tasks:
            options = {f"{t['id']} – {t['title']}": t for t in tasks}
            selection = st.selectbox("Select a task", list(options.keys()))
            selected_task = options[selection]
            with st.form("update_task_form"):
                title_upd = st.text_input("Title", value=selected_task.get("title", ""))
                client_upd = st.text_input("Client Name", value=selected_task.get("client_name") or "")
                service_upd = st.text_input("Service Type", value=selected_task.get("service_type") or "")
                period_upd = st.text_input("Period", value=selected_task.get("period") or "")
                due_upd = st.date_input(
                    "Due Date",
                    value=datetime.date.fromisoformat(selected_task.get("due_date")),
                    min_value=datetime.date.today(),
                )
                priority_upd = st.selectbox(
                    "Priority",
                    [p.value.capitalize() for p in Priority],
                    index=list(Priority).index(Priority(selected_task.get("priority", "medium"))),
                )
                recurrence_upd = st.selectbox(
                    "Recurrence",
                    [r.value.capitalize() for r in Recurrence],
                    index=list(Recurrence).index(Recurrence(selected_task.get("recurrence", "none"))),
                )
                regulatory_upd = st.text_input("Regulatory Reference", value=selected_task.get("regulatory_reference") or "")
                notes_upd = st.text_area("Notes", value=selected_task.get("notes") or "")
                submit_upd = st.form_submit_button("Update")
                if submit_upd:
                    try:
                        updates = {
                            "title": title_upd.strip(),
                            "client_name": client_upd.strip() or None,
                            "service_type": service_upd.strip() or None,
                            "period": period_upd.strip() or None,
                            "due_date": due_upd,
                            "priority": priority_upd.lower(),
                            "recurrence": recurrence_upd.lower(),
                            "regulatory_reference": regulatory_upd.strip() or None,
                            "notes": notes_upd.strip() or None,
                        }
                        service.update_task(selected_task["id"], updates)
                        st.success("Task updated.")
                        st.experimental_rerun()
                    except Exception as e:
                        st.error(str(e))